<?php
/**
 * سكربت الكرون:
 * - يجلب أحدث الفيديوهات من قنوات يوتيوب المخزنة في قاعدة البيانات (channels).
 * - ينشر الفيديوهات الجديدة في قناة تليجرام باستخدام إعدادات (settings).
 * - يمنع التكرار عبر جدول posted_videos.
 * - يراعي الفاصل الزمني min_interval_seconds بإستخدام setting last_run_ts.
 */

require_once __DIR__ . '/../db.php';

date_default_timezone_set(get_setting('timezone', date_default_timezone_get()));

/**
 * جلب إعدادات أساسية
 */
$telegramBotToken = get_setting('telegram_bot_token', '');
$telegramChatId   = get_setting('telegram_chat_id', '');
$minInterval      = (int)get_setting('min_interval_seconds', 900);
$maxPerChannel    = (int)get_setting('max_videos_per_run_per_channel', 3);
$maxPerChannel    = $maxPerChannel > 0 ? $maxPerChannel : 3;

if (!$telegramBotToken || !$telegramChatId) {
    app_log_error("Telegram settings not configured (bot token / chat id).");
    exit(0);
}

$pdo = db();

/**
 * التحقق من الفاصل الزمني
 */
$lastRunTs = (int)get_setting('last_run_ts', 0);
if ($minInterval > 0 && $lastRunTs > 0) {
    $diff = time() - $lastRunTs;
    if ($diff < $minInterval) {
        // لم يحن الوقت بعد
        exit(0);
    }
}

/**
 * جلب القنوات المفعّلة
 */
$channelsStmt = $pdo->query("SELECT * FROM channels WHERE enabled = 1 ORDER BY id ASC");
$channels = $channelsStmt->fetchAll();
if (!$channels) {
    app_log_error("No enabled channels found.");
    // تحديث وقت التشغيل حتى لا يكرر المحاولة بكثافة
    set_setting('last_run_ts', (string)time());
    exit(0);
}

/**
 * دوال HTTP و تليجرام و RSS
 */
function http_get($url)
{
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => 20,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($body === false) {
        throw new RuntimeException("HTTP Error: $err");
    }
    if ($code < 200 || $code >= 300) {
        throw new RuntimeException("HTTP Status $code from $url");
    }

    return $body;
}

function get_feed_url_for_channel($channel)
{
    if (!empty($channel['youtube_feed_url'])) {
        return $channel['youtube_feed_url'];
    }

    $url = $channel['youtube_channel_url'] ?? '';
    if (!$url) {
        throw new RuntimeException("Channel #{$channel['id']} has no URL.");
    }

    $parts = parse_url($url);
    if (!isset($parts['path'])) {
        throw new RuntimeException("Invalid YouTube channel URL: $url");
    }

    if (preg_match('#^/channel/([^/]+)#', $parts['path'], $m)) {
        $channelId = $m[1];
        return 'https://www.youtube.com/feeds/videos.xml?channel_id=' . $channelId;
    }

    throw new RuntimeException("Unable to derive RSS from URL: $url. Please set youtube_feed_url in DB.");
}

function send_to_telegram($botToken, $chatId, $message)
{
    $url = "https://api.telegram.org/bot{$botToken}/sendMessage";

    $payload = [
        'chat_id'                  => $chatId,
        'text'                     => $message,
        'parse_mode'               => 'HTML',
        'disable_web_page_preview' => false,
    ];

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
        CURLOPT_POSTFIELDS     => json_encode($payload),
        CURLOPT_TIMEOUT        => 20,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    $response = curl_exec($ch);
    $err      = curl_error($ch);
    $code     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($response === false) {
        throw new RuntimeException("Telegram HTTP error: $err");
    }

    if ($code < 200 || $code >= 300) {
        throw new RuntimeException("Telegram HTTP status $code. Response: $response");
    }

    $data = json_decode($response, true);
    if (!isset($data['ok']) || !$data['ok']) {
        throw new RuntimeException("Telegram returned error: " . $response);
    }
}

function build_telegram_message($videoUrl, $channelTitle, $videoTitle, $description)
{
    $fullTitle = "📺 {$channelTitle}\n" . $videoTitle;
    $fullDesc  = trim($fullTitle . "\n\n" . $description);

    $fullDesc = htmlspecialchars($fullDesc, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    $safeUrl  = htmlspecialchars($videoUrl, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');

    $maxLen = 3800;
    if (mb_strlen($fullDesc, 'UTF-8') > $maxLen) {
        $fullDesc = mb_substr($fullDesc, 0, $maxLen, 'UTF-8') . '…';
    }

    $linkText = '🎥 مشاهدة الفيديو على يوتيوب';
    $link     = '<a href="' . $safeUrl . '">' . $linkText . '</a>';

    $message = $link . "\n\n" . $fullDesc;

    return $message;
}

/**
 * تنفيذ المنطق لكل قناة
 */

try {
    foreach ($channels as $channel) {
        $channelTitle = $channel['title'] ?: ('Channel #' . $channel['id']);

        try {
            $feedUrl = get_feed_url_for_channel($channel);
            $xmlString = http_get($feedUrl);
            $xml = @simplexml_load_string($xmlString);
            if ($xml === false) {
                throw new RuntimeException("Failed to parse XML feed for channel {$channelTitle}");
            }

            $namespaces = $xml->getNamespaces(true);
            $nsMedia = $namespaces['media'] ?? null;
            $nsYt    = $namespaces['yt'] ?? null;

            $newCount = 0;

            foreach ($xml->entry as $entry) {
                if ($newCount >= $maxPerChannel) {
                    break;
                }

                // videoId
                $videoId = '';
                if ($nsYt) {
                    $yt = $entry->children($nsYt);
                    $videoId = (string)$yt->videoId;
                } else {
                    $videoLink = (string)$entry->link['href'];
                    parse_str(parse_url($videoLink, PHP_URL_QUERY), $q);
                    $videoId = $q['v'] ?? '';
                }

                if (!$videoId) {
                    continue;
                }

                // تحقق من التكرار
                $checkStmt = $pdo->prepare("SELECT id FROM posted_videos WHERE video_id = ? AND channel_id = ? LIMIT 1");
                $checkStmt->execute([$videoId, $channel['id']]);
                if ($checkStmt->fetchColumn()) {
                    continue; // سبق نشره
                }

                $title    = (string)$entry->title;
                $videoUrl = (string)$entry->link['href'];

                $description = '';
                if ($nsMedia) {
                    $media = $entry->children($nsMedia);
                    if (isset($media->group->description)) {
                        $description = (string)$media->group->description;
                    }
                }

                $message = build_telegram_message($videoUrl, $channelTitle, $title, $description);
                send_to_telegram($telegramBotToken, $telegramChatId, $message);

                // حفظ في posted_videos
                $insertStmt = $pdo->prepare("INSERT INTO posted_videos (video_id, channel_id, posted_at) VALUES (?, ?, NOW())");
                $insertStmt->execute([$videoId, $channel['id']]);

                $newCount++;
            }

        } catch (Throwable $eChannel) {
            app_log_error("Channel {$channelTitle}: " . $eChannel->getMessage());
            continue;
        }
    }

    // تحديث وقت التشغيل
    set_setting('last_run_ts', (string)time());

} catch (Throwable $e) {
    app_log_error("CRON ERROR: " . $e->getMessage());
    exit(1);
}
